---
title: Overview · Cloudflare Sandbox SDK docs
description: The Sandbox SDK enables you to run untrusted code securely in
  isolated environments. Built on Containers, Sandbox SDK provides a simple API
  for executing commands, managing files, running background processes, and
  exposing services — all from your Workers applications.
lastUpdated: 2025-10-15T17:28:03.000Z
chatbotDeprioritize: false
source_url:
  html: https://developers.cloudflare.com/sandbox/
  md: https://developers.cloudflare.com/sandbox/index.md
---

Build secure, isolated code execution environments

Available on Workers Paid plan

The Sandbox SDK enables you to run untrusted code securely in isolated environments. Built on [Containers](https://developers.cloudflare.com/containers/), Sandbox SDK provides a simple API for executing commands, managing files, running background processes, and exposing services — all from your [Workers](https://developers.cloudflare.com/workers/) applications.

Sandboxes are ideal for building AI agents that need to execute code, interactive development environments, data analysis platforms, CI/CD systems, and any application that needs secure code execution at the edge. Each sandbox runs in its own isolated container with a full Linux environment, providing strong security boundaries while maintaining performance.

With Sandbox, you can execute Python scripts, run Node.js applications, analyze data, compile code, and perform complex computations — all with a simple TypeScript API and no infrastructure to manage.

* Execute Commands

  ```typescript
  import { getSandbox } from '@cloudflare/sandbox';


  export default {
    async fetch(request: Request, env: Env): Promise<Response> {
      const sandbox = getSandbox(env.Sandbox, 'user-123');


      // Execute a command and get the result
      const result = await sandbox.exec('python --version');


      return Response.json({
        output: result.stdout,
        exitCode: result.exitCode,
        success: result.success
      });
    }
  };
  ```

* Code Interpreter

  ```typescript
  import { getSandbox } from '@cloudflare/sandbox';


  export default {
    async fetch(request: Request, env: Env): Promise<Response> {
      const sandbox = getSandbox(env.Sandbox, 'user-123');


      // Create a Python execution context
      const ctx = await sandbox.createCodeContext({ language: 'python' });


      // Execute Python code with automatic result capture
      const result = await sandbox.runCode(`
  import pandas as pd
  data = {'product': ['A', 'B', 'C'], 'sales': [100, 200, 150]}
  df = pd.DataFrame(data)
  df['sales'].sum()  # Last expression is automatically returned
      `, { context: ctx });


      return Response.json({
        result: result.results?.[0]?.text,
        logs: result.logs
      });
    }
  };
  ```

* File Operations

  ```typescript
  import { getSandbox } from '@cloudflare/sandbox';


  export default {
    async fetch(request: Request, env: Env): Promise<Response> {
      const sandbox = getSandbox(env.Sandbox, 'user-123');


      // Create a project structure
      await sandbox.mkdir('/workspace/project/src', { recursive: true });


      // Write files
      await sandbox.writeFile(
        '/workspace/project/package.json',
        JSON.stringify({ name: 'my-app', version: '1.0.0' })
      );


      // Read a file back
      const content = await sandbox.readFile('/workspace/project/package.json');


      return Response.json({ content });
    }
  };
  ```

[Get started](https://developers.cloudflare.com/sandbox/get-started/)

[API Reference](https://developers.cloudflare.com/sandbox/api/)

***

## Features

### Execute commands securely

Run shell commands, Python scripts, Node.js applications, and more in isolated containers with streaming output support and automatic timeout handling.

[Learn about command execution](https://developers.cloudflare.com/sandbox/guides/execute-commands/)

### Manage files and processes

Read, write, and manipulate files in the sandbox filesystem. Run background processes, monitor output, and manage long-running operations.

[Learn about file operations](https://developers.cloudflare.com/sandbox/guides/manage-files/)

### Expose services with preview URLs

Expose HTTP services running in your sandbox with automatically generated preview URLs, perfect for interactive development environments and application hosting.

[Learn about preview URLs](https://developers.cloudflare.com/sandbox/guides/expose-services/)

***

## Use Cases

Build powerful applications with Sandbox:

### AI Code Execution

Execute code generated by Large Language Models safely and reliably. Perfect for AI agents, code assistants, and autonomous systems that need to run untrusted code.

### Data Analysis & Notebooks

Create interactive data analysis environments with Python, pandas, and visualization libraries. Build notebook-like experiences at the edge.

### Interactive Development Environments

Build cloud IDEs, coding playgrounds, and collaborative development tools with full Linux environments and preview URLs.

### CI/CD & Build Systems

Run tests, compile code, and execute build pipelines in isolated environments with parallel execution and streaming logs.

***

## Related products

**[Containers](https://developers.cloudflare.com/containers/)**

Serverless container runtime that powers Sandbox, enabling you to run any containerized workload on the edge.

**[Workers AI](https://developers.cloudflare.com/workers-ai/)**

Run machine learning models and LLMs on the network. Combine with Sandbox for secure AI code execution workflows.

**[Durable Objects](https://developers.cloudflare.com/durable-objects/)**

Stateful coordination layer that enables Sandbox to maintain persistent environments with strong consistency.

***

## More resources

[Examples](https://developers.cloudflare.com/sandbox/examples/)

Explore complete examples including AI code execution, data analysis, and interactive environments.

[Beta Information](https://developers.cloudflare.com/sandbox/platform/beta-info/)

Learn about the Sandbox Beta, current status, and upcoming features.

[Pricing](https://developers.cloudflare.com/sandbox/platform/pricing/)

Understand Sandbox pricing based on the underlying Containers platform.

[Limits](https://developers.cloudflare.com/sandbox/platform/limits/)

Learn about resource limits, quotas, and best practices for working within them.

[Best Practices](https://developers.cloudflare.com/sandbox/best-practices/)

Learn security best practices, performance optimization, and production deployment patterns.

[GitHub Repository](https://github.com/cloudflare/sandbox-sdk)

View the SDK source code, report issues, and contribute to the project.

[Developer Discord](https://discord.cloudflare.com)

Connect with the Workers community on Discord. Ask questions, share what you're building, and get help from other developers.
