---
title: McpAgent — API Reference · Cloudflare Agents docs
description: "When you build MCP Servers on Cloudflare, you extend the McpAgent
  class, from the Agents SDK, like this:"
lastUpdated: 2025-08-15T20:11:52.000Z
chatbotDeprioritize: false
tags: MCP
source_url:
  html: https://developers.cloudflare.com/agents/model-context-protocol/mcp-agent-api/
  md: https://developers.cloudflare.com/agents/model-context-protocol/mcp-agent-api/index.md
---

When you build MCP Servers on Cloudflare, you extend the [`McpAgent` class](https://github.com/cloudflare/agents/blob/5881c5d23a7f4580600029f69307cfc94743e6b8/packages/agents/src/mcp.ts), from the Agents SDK, like this:

* JavaScript

  ```js
  import { McpAgent } from "agents/mcp";
  import { McpServer } from "@modelcontextprotocol/sdk/server/mcp.js";
  import { z } from "zod";


  export class MyMCP extends McpAgent {
    server = new McpServer({ name: "Demo", version: "1.0.0" });


    async init() {
      this.server.tool(
        "add",
        { a: z.number(), b: z.number() },
        async ({ a, b }) => ({
          content: [{ type: "text", text: String(a + b) }],
        }),
      );
    }
  }
  ```

* TypeScript

  ```ts
  import { McpAgent } from "agents/mcp";
  import { McpServer } from "@modelcontextprotocol/sdk/server/mcp.js";
  import { z } from "zod";


  export class MyMCP extends McpAgent {
    server = new McpServer({ name: "Demo", version: "1.0.0" });


    async init() {
      this.server.tool(
        "add",
        { a: z.number(), b: z.number() },
        async ({ a, b }) => ({
          content: [{ type: "text", text: String(a + b) }],
        }),
      );
    }
  }
  ```

This means that each instance of your MCP server has its own durable state, backed by a [Durable Object](https://developers.cloudflare.com/durable-objects/), with its own [SQL database](https://developers.cloudflare.com/agents/api-reference/store-and-sync-state).

Your MCP server doesn't necessarily have to be an Agent. You can build MCP servers that are stateless, and just add [tools](https://developers.cloudflare.com/agents/model-context-protocol/tools) to your MCP server using the `@modelcontextprotocol/typescript-sdk` package.

But if you want your MCP server to:

* remember previous tool calls, and responses it provided
* provide a game to the MCP client, remembering the state of the game board, previous moves, and the score
* cache the state of a previous external API call, so that subsequent tool calls can reuse it
* do anything that an Agent can do, but allow MCP clients to communicate with it

You can use the APIs below in order to do so.

#### Hibernation Support

`McpAgent` instances automatically support [WebSockets Hibernation](https://developers.cloudflare.com/durable-objects/best-practices/websockets/#websocket-hibernation-api), allowing stateful MCP servers to sleep during inactive periods while preserving their state. This means your agents only consume compute resources when actively processing requests, optimizing costs while maintaining the full context and conversation history.

Hibernation is enabled by default and requires no additional configuration.

#### Authentication & Authorization

The McpAgent class provides seamless integration with the [OAuth Provider Library](https://github.com/cloudflare/workers-oauth-provider) for [authentication and authorization](https://developers.cloudflare.com/agents/model-context-protocol/authorization/).

When a user authenticates to your MCP server, their identity information and tokens are made available through the `props` parameter, allowing you to:

* access user-specific data
* check user permissions before performing operations
* customize responses based on user attributes
* use authentication tokens to make requests to external services on behalf of the user

### State synchronization APIs

The `McpAgent` class makes the following subset of methods from the [Agents SDK](https://developers.cloudflare.com/agents/api-reference/agents-api/) available:

* [`state`](https://developers.cloudflare.com/agents/api-reference/store-and-sync-state/)
* [`initialState`](https://developers.cloudflare.com/agents/api-reference/store-and-sync-state/#set-the-initial-state-for-an-agent)
* [`setState`](https://developers.cloudflare.com/agents/api-reference/store-and-sync-state/)
* [`onStateUpdate`](https://developers.cloudflare.com/agents/api-reference/store-and-sync-state/#synchronizing-state)
* [`sql`](https://developers.cloudflare.com/agents/api-reference/agents-api/#sql-api)

State resets after the session ends

Currently, each client session is backed by an instance of the `McpAgent` class. This is handled automatically for you, as shown in the [getting started guide](https://developers.cloudflare.com/agents/guides/remote-mcp-server). This means that when the same client reconnects, they will start a new session, and the state will be reset.

For example, the following code implements an MCP server that remembers a counter value, and updates the counter when the `add` tool is called:

* JavaScript

  ```js
  import { McpAgent } from "agents/mcp";
  import { McpServer } from "@modelcontextprotocol/sdk/server/mcp.js";
  import { z } from "zod";


  export class MyMCP extends McpAgent {
    server = new McpServer({
      name: "Demo",
      version: "1.0.0",
    });


    initialState = {
      counter: 1,
    };


    async init() {
      this.server.resource(`counter`, `mcp://resource/counter`, (uri) => {
        return {
          contents: [{ uri: uri.href, text: String(this.state.counter) }],
        };
      });


      this.server.tool(
        "add",
        "Add to the counter, stored in the MCP",
        { a: z.number() },
        async ({ a }) => {
          this.setState({ ...this.state, counter: this.state.counter + a });


          return {
            content: [
              {
                type: "text",
                text: String(`Added ${a}, total is now ${this.state.counter}`),
              },
            ],
          };
        },
      );
    }


    onStateUpdate(state) {
      console.log({ stateUpdate: state });
    }
  }
  ```

* TypeScript

  ```ts
  import { McpAgent } from "agents/mcp";
  import { McpServer } from "@modelcontextprotocol/sdk/server/mcp.js";
  import { z } from "zod";


  type State = { counter: number };


  export class MyMCP extends McpAgent<Env, State, {}> {
    server = new McpServer({
      name: "Demo",
      version: "1.0.0",
    });


    initialState: State = {
      counter: 1,
    };


    async init() {
      this.server.resource(`counter`, `mcp://resource/counter`, (uri) => {
        return {
          contents: [{ uri: uri.href, text: String(this.state.counter) }],
        };
      });


      this.server.tool(
        "add",
        "Add to the counter, stored in the MCP",
        { a: z.number() },
        async ({ a }) => {
          this.setState({ ...this.state, counter: this.state.counter + a });


          return {
            content: [
              {
                type: "text",
                text: String(`Added ${a}, total is now ${this.state.counter}`),
              },
            ],
          };
        },
      );
    }


    onStateUpdate(state: State) {
      console.log({ stateUpdate: state });
    }
  }
  ```

### Not yet supported APIs

The following APIs from the Agents SDK are not yet available on `McpAgent`:

* [WebSocket APIs](https://developers.cloudflare.com/agents/api-reference/websockets/) (`onMessage`, `onError`, `onClose`, `onConnect`)
* [Scheduling APIs](https://developers.cloudflare.com/agents/api-reference/schedule-tasks/) `this.schedule`
