---
title: Execute commands · Cloudflare Sandbox SDK docs
description: This guide shows you how to execute commands in the sandbox, handle
  output, and manage errors effectively.
lastUpdated: 2025-10-15T15:03:46.000Z
chatbotDeprioritize: false
source_url:
  html: https://developers.cloudflare.com/sandbox/guides/execute-commands/
  md: https://developers.cloudflare.com/sandbox/guides/execute-commands/index.md
---

This guide shows you how to execute commands in the sandbox, handle output, and manage errors effectively.

## Choose the right method

The SDK provides two methods for command execution:

* **`exec()`** - Run a command and wait for complete result. Best for most use cases.
* **`execStream()`** - Stream output in real-time. Best for long-running commands where you need immediate feedback.

## Execute basic commands

Use `exec()` for simple commands that complete quickly:

* JavaScript

  ```js
  import { getSandbox } from "@cloudflare/sandbox";


  const sandbox = getSandbox(env.Sandbox, "my-sandbox");


  // Execute a single command
  const result = await sandbox.exec("python --version");


  console.log(result.stdout); // "Python 3.11.0"
  console.log(result.exitCode); // 0
  console.log(result.success); // true
  ```

* TypeScript

  ```ts
  import { getSandbox } from '@cloudflare/sandbox';


  const sandbox = getSandbox(env.Sandbox, 'my-sandbox');


  // Execute a single command
  const result = await sandbox.exec('python --version');


  console.log(result.stdout);   // "Python 3.11.0"
  console.log(result.exitCode); // 0
  console.log(result.success);  // true
  ```

## Pass arguments safely

When passing user input or dynamic values, avoid string interpolation to prevent injection attacks:

* JavaScript

  ```js
  // Unsafe - vulnerable to injection
  const filename = userInput;
  await sandbox.exec(`cat ${filename}`);


  // Safe - use proper escaping or validation
  const safeFilename = filename.replace(/[^a-zA-Z0-9_.-]/g, "");
  await sandbox.exec(`cat ${safeFilename}`);


  // Better - write to file and execute
  await sandbox.writeFile("/tmp/input.txt", userInput);
  await sandbox.exec("python process.py /tmp/input.txt");
  ```

* TypeScript

  ```ts
  // Unsafe - vulnerable to injection
  const filename = userInput;
  await sandbox.exec(`cat ${filename}`);


  // Safe - use proper escaping or validation
  const safeFilename = filename.replace(/[^a-zA-Z0-9_.-]/g, '');
  await sandbox.exec(`cat ${safeFilename}`);


  // Better - write to file and execute
  await sandbox.writeFile('/tmp/input.txt', userInput);
  await sandbox.exec('python process.py /tmp/input.txt');
  ```

## Handle errors

Commands can fail in two ways:

1. **Non-zero exit code** - Command ran but failed (result.success === false)
2. **Execution error** - Command couldn't start (throws exception)

* JavaScript

  ```js
  try {
    const result = await sandbox.exec("python analyze.py");


    if (!result.success) {
      // Command failed (non-zero exit code)
      console.error("Analysis failed:", result.stderr);
      console.log("Exit code:", result.exitCode);


      // Handle specific exit codes
      if (result.exitCode === 1) {
        throw new Error("Invalid input data");
      } else if (result.exitCode === 2) {
        throw new Error("Missing dependencies");
      }
    }


    // Success - process output
    return JSON.parse(result.stdout);
  } catch (error) {
    // Execution error (couldn't start command)
    console.error("Execution failed:", error.message);
    throw error;
  }
  ```

* TypeScript

  ```ts
  try {
    const result = await sandbox.exec('python analyze.py');


    if (!result.success) {
      // Command failed (non-zero exit code)
      console.error('Analysis failed:', result.stderr);
      console.log('Exit code:', result.exitCode);


      // Handle specific exit codes
      if (result.exitCode === 1) {
        throw new Error('Invalid input data');
      } else if (result.exitCode === 2) {
        throw new Error('Missing dependencies');
      }
    }


    // Success - process output
    return JSON.parse(result.stdout);


  } catch (error) {
    // Execution error (couldn't start command)
    console.error('Execution failed:', error.message);
    throw error;
  }
  ```

## Execute shell commands

The sandbox supports shell features like pipes, redirects, and chaining:

* JavaScript

  ```js
  // Pipes and filters
  const result = await sandbox.exec('ls -la | grep ".py" | wc -l');
  console.log("Python files:", result.stdout.trim());


  // Output redirection
  await sandbox.exec("python generate.py > output.txt 2> errors.txt");


  // Multiple commands
  await sandbox.exec("cd /workspace && npm install && npm test");
  ```

* TypeScript

  ```ts
  // Pipes and filters
  const result = await sandbox.exec('ls -la | grep ".py" | wc -l');
  console.log('Python files:', result.stdout.trim());


  // Output redirection
  await sandbox.exec('python generate.py > output.txt 2> errors.txt');


  // Multiple commands
  await sandbox.exec('cd /workspace && npm install && npm test');
  ```

## Execute Python scripts

* JavaScript

  ```js
  // Run inline Python
  const result = await sandbox.exec('python -c "print(sum([1, 2, 3, 4, 5]))"');
  console.log("Sum:", result.stdout.trim()); // "15"


  // Run a script file
  await sandbox.writeFile(
    "/workspace/analyze.py",
    `
  import sys
  print(f"Argument: {sys.argv[1]}")
  `,
  );


  await sandbox.exec("python /workspace/analyze.py data.csv");
  ```

* TypeScript

  ```ts
  // Run inline Python
  const result = await sandbox.exec('python -c "print(sum([1, 2, 3, 4, 5]))"');
  console.log('Sum:', result.stdout.trim()); // "15"


  // Run a script file
  await sandbox.writeFile('/workspace/analyze.py', `
  import sys
  print(f"Argument: {sys.argv[1]}")
  `);


  await sandbox.exec('python /workspace/analyze.py data.csv');
  ```

## Best practices

* **Check exit codes** - Always verify `result.success` and `result.exitCode`
* **Validate inputs** - Escape or validate user input to prevent injection
* **Use streaming** - For long operations, use `execStream()` for real-time feedback
* **Handle errors** - Check stderr for error details

## Troubleshooting

### Command not found

Verify the command exists in the container:

* JavaScript

  ```js
  const check = await sandbox.exec("which python3");
  if (!check.success) {
    console.error("python3 not found");
  }
  ```

* TypeScript

  ```ts
  const check = await sandbox.exec('which python3');
  if (!check.success) {
    console.error('python3 not found');
  }
  ```

### Working directory issues

Use absolute paths or change directory:

* JavaScript

  ```js
  // Use absolute path
  await sandbox.exec("python /workspace/my-app/script.py");


  // Or change directory
  await sandbox.exec("cd /workspace/my-app && python script.py");
  ```

* TypeScript

  ```ts
  // Use absolute path
  await sandbox.exec('python /workspace/my-app/script.py');


  // Or change directory
  await sandbox.exec('cd /workspace/my-app && python script.py');
  ```

## Related resources

* [Commands API reference](https://developers.cloudflare.com/sandbox/api/commands/) - Complete method documentation
* [Background processes guide](https://developers.cloudflare.com/sandbox/guides/background-processes/) - Managing long-running processes
* [Streaming output guide](https://developers.cloudflare.com/sandbox/guides/streaming-output/) - Advanced streaming patterns
* [Code Interpreter guide](https://developers.cloudflare.com/sandbox/guides/code-execution/) - Higher-level code execution
